# silen v0.2 (stable)
import pygame, sys, os, re, math
from extremes import uMax,uMin
from pygame.locals import *
class CoreVar:
    """a non-updated animated value optimization. faster calcs and quick creation
__init__(ftime,dist,animtype)
    ftime    = total number of frames
    dist     = final value (start value is 0)
    animtype = 0 [+ 1 for smooth start][+ 2 for smooth end]
                 [+ 3 for trigonometric smoothness]
instance(frame=0) or instance[frame=0]
    frame = the current frame in the animation (between 0 and dist initialized)
"""
    def __init__(self,ftime,dist,animtype):
        self.dist = dist; self.ftime = ftime
        self.lambda1 = self.lambda2 = self.crethlp = None #for coverity, default the possibly undefined attributes
        if animtype == 0: #linear
            self.famt = self.dist/float(ftime)
            self.callret = lambda frame: frame*self.famt
        elif animtype == 1: #smooth start
            self.famt = self.dist/float(ftime**2)
            self.callret = lambda frame: (frame**2)*self.famt
        elif animtype == 2: #smooth end
            self.famt = -self.dist/float(ftime**2)
            self.callret = lambda frame: self.famt*((frame-self.ftime)**2)+self.dist
        elif animtype == 3: #smooth start & end
            self.famt = 2*self.dist/float(ftime**2)
            self.lambda1 = lambda frame: (frame**2)*self.famt
            self.lambda2 = lambda frame: -self.famt*((frame-self.ftime)**2)+self.dist
            self.crethlp = lambda frame: (frame <= self.ftime/2) and self.lambda1 or self.lambda2 #lambda1 is an object, so never 0
            self.callret = lambda frame: self.crethlp(frame)(frame)
        elif animtype == 4: #trig-based smooth start
            self.famt = self.dist
            self.ftime = math.pi/float(2*ftime)
            self.callret = lambda frame: self.famt*(1-math.cos(frame*self.ftime))
        elif animtype == 5: #trig-based smooth end
            self.famt = self.dist
            self.ftime = math.pi/float(2*ftime)
            self.callret = lambda frame: self.famt*math.sin(frame*self.ftime)
        elif animtype == 6: #trig-based smooth start & end
            self.famt = self.dist/2.0
            self.ftime = math.pi/float(ftime)
            self.callret = lambda frame: self.famt*(1-math.cos(frame*self.ftime))
    def __call__(self,frame=0): return self.callret(frame)
    def __getitem__(self,frame=0): return self.callret(frame)
class LinCtr:
    """init([maxix,minix=0,startix=0,startdir=1],[bounce=F,repeat=T,active=T])
an updated linear counter - works with a CoreVar. update once per frame.
__init__(posargs,ctrlargs=[])
    posargs = [maxix,startix=0,startdir=1,minix=0]
        index range = [minix,maxix) w/o bounce, [minix,maxix] w/ bounce
        startix = start index (can be a fraction)
        startdir = increment value (usually use +1 or -1, or 0 to pause)
    ctrlargs = [bounce=False,repeat=False,active=True]
        bounce: change dir when pos is >= max (or <= min with repeat on)
        repeat: with bounce off, when pos >= max, go to min and continue
        active: self-maintained var indicating when to update LinCtr
update(timestep=1.0) #call once per frame
    timestep = amount of time (in frames) that has passed since last update
instance() returns the current index value. no computation cost.
"""
    def __init__(self,posargs,ctrlargs=[]): self.posargsunpack(*posargs); self.ctrlargsunpack(*ctrlargs)
    def posargsunpack(self,maxix,minix=0,startix=0,startdir=1): self.max = maxix; self.min = minix; self.pos = startix; self.dir = startdir
    def ctrlargsunpack(self,bounce=False,repeat=False,active=True): self.bounce = bounce; self.repeat = repeat; self.active = active
    #def oneloop(self): return bounce and (maxix-minix)*2 or (maxix-minix)
    def update(self,timestep=1.0):
        if self.active:
            self.pos += timestep*self.dir
            if self.pos >= self.max:
                self.pos = self.max
                if self.bounce: self.dir = -1
                elif self.repeat: self.pos = 0
                else: self.active = False
            if self.pos <= self.min:
                self.pos = self.min
                if self.repeat: self.dir = 1
                else: self.active = False
    def __call__(self): return self.pos
class AnimVar:
    """([ftime,ix,dir,minix],dist,animtype,[b=F,r=T,a=T],[delay,tie,looptype])
an updated handler for a CoreVar and a LinCtr. update each frame.
__init__(posargs,dist,animtype,ctrlargs=[],loopargs=[])
    posargs = [ftime,startix=0,startdir=1,minix=0]
        (args are same as LinCtr posargs)
    dist = final value (start value is 0)
    animtype = 0 [+ 1 for smooth start][+ 2 for smooth end]
                 [+ 3 for trigonometric smoothness]
    ctrlargs = [bounce=False,repeat=False,active=True]
        (args are same as LinCtr ctrlargs)
    loopargs = [delay=0.0,tie=None,looptype=0]
        delay = amount of frames to wait before resuming update execution.
        tie = an AnimVar, Var, or Sprite, which prevents update until inactive.
        looptype = 0 (usually - for vars that loop internally)
                 = 1 (for the first var in a sequence of looping vars)
                     (this var can't have ftime==1, or looping will break)
                 = 2+(for the remaining vars in a sequence of looping vars)
                 = -1(for a var that runs when another var IS active)
update(timestep=1.0) #call once per frame
    timestep = amount of time (in frames) that has passed since last update
instance() returns the current index value. no computation cost.
active() returns True if the current var is running (loop seqs are always True)
"""
    def __init__(self,posargs,dist,animtype,ctrlargs=[],loopargs=[]):
        self.data = CoreVar(posargs[0],dist,animtype); self.cframe = LinCtr(posargs,ctrlargs)
        self.offset = dist; self.loopargsunpack(posargs,ctrlargs,*loopargs)
    def loopargsunpack(self,posargs,ctrlargs,delay=0.0,tie=None,looptype=0):
        self.delay = delay; self.tie = tie #the higher the var level, the more dangerous it is to tie.
        self.update = looptype >= 0 and getattr(self,"update%d" % min(2,looptype)) or self.updatealt #faster to use min than default condition in getattr
        if looptype > 0: self.cframe.active = False; self.remdelay = delay; self.cframeargs = (posargs,ctrlargs)
    def loopto(self,tie): self.tie = tie #the first var in the loop should tie to the last var in the loop (after creation)
    def update0(self,timestep=1.0): #standard update method
        if self.tie: #until the tied var is finished, this var waits
            if self.tie.active(): return
            self.tie = None
        if self.delay != 0.0: self.delay = max(0.0,self.delay - timestep); return
        self.cframe.update(timestep)
    def update1(self,timestep=1.0): #update for the first in a looped series
        if not self.cframe.active: #var re-activates when tie is stopped
            if self.tie.active(): return
            self.cframe = LinCtr(*self.cframeargs); self.cframe.active = True; self.delay = self.remdelay
        if self.delay != 0.0: self.delay = max(0.0,self.delay - timestep); return
        self.cframe.update(timestep)
    def update2(self,timestep=1.0): #update for the rest in a looped series
        if not self.cframe.active: #var re-activates when tie is running
            if not self.tie.active(): return
            self.cframe = LinCtr(*self.cframeargs); self.cframe.active = True; self.delay = self.remdelay
        if self.tie.active(): return #var waits until tie is finished
        if self.delay != 0.0: self.delay = max(0.0,self.delay - timestep); return
        self.cframe.update(timestep)
    def updatealt(self,timestep=1.0): #standard update method, but waits for tie to be True
        if self.tie: #until the tied var is active, this var waits
            if not self.tie.active(): return
            self.tie = None
        if self.delay != 0.0: self.delay = max(0.0,self.delay - timestep); return
        self.cframe.update(timestep)
    def __call__(self): return self.data[self.cframe()]
    def addend(self): return not self.cframe.bounce or self.cframe.repeat #bounce=T & repeat=F doesn't go anywhere...
    def active(self): return self.cframe.active
class Var:
    """main variable class. handles all single AnimVars of a particular var.
__init__(start=0,lbound=uMin,ubound=uMax)
    start = the start offset of the variable
    lbound, ubound = lower/upper bound for the var's values
add(posargs,dist,animtype=0,ctrlargs=[],loopargs=[]) #checks for ftime==0 or dist==0
    posargs = [ftime,minix=0,startix=0,startdir=1]
        (args are same as LinCtr posargs)
    dist = final distance (after ftime frames)
    animtype = 0 [+ 1 for smooth start][+ 2 for smooth end]
                 [+ 3 for trigonometric smoothness]
    ctrlargs = [bounce=False,repeat=False,active=True]
        (args are same as LinCtr ctrlargs)
    loopargs = [delay=0.0,tie=None,looptype=0]
        (args are same as AnimVar loopargs)
cut(animvar=None) #cut an animation short (as much as it has already animated stays)
end(animvar=None) #finish an animation instantly (jumps to end immediately)
    animvar = an AnimVar object in the var (must exist)
update(timestep=1.0) #call once per frame
    timestep = amount of time (in frames) that has passed since last update
final() #find the final value of the Var (once all AnimVars finish)
active() returns True if the current var is running (loop seqs are always True)
"""
    def __init__(self,start=0,lbound=uMin,ubound=uMax): self.offset = start; self.lb = lbound; self.ub = ubound; self.data = []
    def add(self,posargs,dist,animtype=0,ctrlargs=[],loopargs=[]): #returns AnimVar for using in ties
        if not posargs[0] or not dist: return None
        coreadd = AnimVar(posargs,dist,animtype,ctrlargs,loopargs); self.data.append(coreadd); return coreadd
    def movto(self,posargs,fval,animtype=0,ctrlargs=[],loopargs=[]): #returns AnimVar for using in ties - like add but moves to absolute position
        return self.add(posargs,fval-self.rawfinal(),animtype,ctrlargs,loopargs)
    def cut(self,animvar=None):
        if animvar: self.offset += self.data.pop(self.data.index(animvar))() #stop one AnimVar only (by 'name')
        else: self.offset = max(self.lb,min(self.ub,sum([av() for av in self.data],self.offset))); self.data = [] #stop all AnimVars
    def end(self,animvar=None):
        if animvar: self.offset += animvar.addend() and self.data.pop(self.data.index(animvar)).offset or 0 #end one AnimVar only (by 'name')
        else: self.offset = max(self.lb,min(self.ub,sum([av.offset for av in self.data if av.addend()],self.offset))); self.data = [] #end all AnimVars
    def final(self): return max(self.lb,min(self.ub,sum([av.offset for av in self.data if av.addend()],self.offset)))
    def rawfinal(self): return sum([av.offset for av in self.data if av.addend()],self.offset) #no clipping, for calculations that need the real final value
    def __call__(self): return max(self.lb,min(self.ub,sum([av() for av in self.data],self.offset)))
    def update(self,timestep=1.0):
        for av in self.data:
            av.update(timestep)
            if not av.active() and av.addend(): self.offset += av.offset
        self.data = [av for av in self.data if av.active()]
    def active(self): return bool(len(self.data))
class PicFrame:
    """(times,[(w,h),numwide,numhigh,wrap=T],[ix,minix,maxix,endix,timer,dir],a=T)
manages an animated sprite sequence in a larger image.
__init__(timelist,seqargs,animargs,active=True)
    timelist = list of delays at each position in the image sequence
    seqargs = [whtuple,numwide=1,numhigh=1,wrap=True]
        whtuple = (width,height) of the full image in pixels
        numwide,numhigh = number of animated frames the image holds
        wrap = continue animating from the last frame to the first
    animargs = [startix=0,minix=0,maxix=None,endix=None,starttimer=0,
                animdir=1,active=None(no change)]
        startix = frame to start animating on
        animation range = [minix,maxix) (like range(minix,maxix))
        endix = frame to stop animating on
        starttimer = initial head-start on the first delay
    active = whether to animate or not (usually True)
update(timestep=1.0) #call once per frame
    timestep = amount of time (in frames) that has passed since last update
instance() #get the rect for the image to display (within the larger image)
"""
    def __init__(self,timelist,seqargs,animargs=[],active=True):
        self.list = timelist; self.seqargsunpack(*seqargs); self.setanimrange(*animargs); self.active = active
    def seqargsunpack(self,whtuple,numwide=1,numhigh=1,wrap=True):
        self.width = int(whtuple[0]/numwide); self.height = int(whtuple[1]/numhigh); self.numwide = int(numwide); self.wrap = wrap
    def setanimrange(self,startix=0,minix=0,maxix=None,endix=None,starttimer=0,animdir=1,active=None):
        self.ix = int(startix); self.minix = minix; self.maxix = (maxix == None) and len(self.list) or maxix
        self.endix = endix; self.timer = starttimer; self.dir = animdir
        if active != None: self.active = active
    def update(self,timestep=1.0): #call once per frame, pass amt of time (in frames) that has passed
        if self.active:
            self.timer += timestep
            if self.timer >= self.list[self.ix]:
                self.timer = 0; self.ix += self.dir
                if self.ix == self.maxix:
                    if self.wrap: self.ix = self.minix
                    else: self.active = False
                if self.ix == self.endix: self.active = False #this way, None works, and a complete cycle can occur if startix == endix
    def __call__(self): #get the rect for the image to display (within the larger image)
        return Rect(self.width*(self.ix%self.numwide),self.height*(self.ix//self.numwide),self.width,self.height)
class Sprite:
    """
#add support for semi-transparent overlays on images, borders on shapes, clipping (see surface clipping rects), and open up special surface flag args to interface
__init__(stype,arglist,cpostuple)
    stype = 0 for default image-based sprites with colorkeys
            1 for image-based sprites with full transparency
            2 is for image-based sprites pre-loaded in surfaces
            3 for solids
            4 for circles (limited to min(width,height))
    arglist is different for each stype
        stype 0:   [relpath,colorkey=None,alphastart=255]
        stype 1:   [relpath]
        stype 2:   [surface]
        stype 3,4: [whtuple,fillcolor]
            relpath = 'relative_path/filename.ext'
            colorkey = (R,G,B), or None for no transparent color
            alphastart = starting transparency for the full image
            fillcolor = (R,G,B[,A])
"""
    timestep = 1.0
    def __init__(self,stype,arglist,cpostuple):
        self.alpha = Var(255,0,255); getattr(self,"init%d" % stype)(*arglist)
        self.cx = Var(cpostuple[0]); self.cy = Var(cpostuple[1]); self.xscale = Var(1.0); self.yscale = Var(1.0); self.rot = Var(0.0)
        self.fill = lambda: (self.rfill(),self.gfill(),self.bfill(),self.afill()) #afill only works on surfaces w/ alpha
        self.svlist = [self.cx,self.cy,self.xscale,self.yscale,self.rot,self.alpha,self.rfill,self.gfill,self.bfill,self.afill]
        self.data = None; self.rect = None; self.pframe = None
        self.oldxscale = 1.0; self.oldyscale = 1.0; self.oldrot = 0.0; self.oldalpha = 255; self.oldpframe = None
        (self.oldrfill,self.oldgfill,self.oldbfill,self.oldafill) = self.oldfill = (0,0,0,0) #same ref, meaning changing one will affect the other (as it should)
        self.changepic(True)
    def init0(self,relpath,colorkey=None,alphastart=255): #colorkey image sprites
        self.alldata = pygame.image.load(relpath).convert()
        self.alldata.set_colorkey(colorkey)
        self.alpha.offset = max(0,min(255,alphastart)) #only colorkeys support full image opacity
        self.setfill(0,0,0,0) #only fully opaque overlays on colorkey pictures are supported
    def init1(self,relpath): #alpha channel image sprites
        self.alldata = pygame.image.load(relpath).convert_alpha()
        self.setfill(0,0,0,0) #semi-transparent overlays on pictures are supported
    def init2(self,surface): #image sprites with preset surfaces
        self.alldata = surface
        self.setfill(0,0,0,0) #semi-transparent overlays on pictures are supported
    def init3(self,whtuple,fillcolor): #solid fill sprites
        self.alldata = pygame.Surface(whtuple).convert_alpha()
        self.setfill(*fillcolor) #semi-transparent fills are supported, (R,G,B) only colors are fully opaque by default.
    def init4(self,whtuple,fillcolor): #solid circle/oval the size of the surface
        self.alldata = pygame.Surface(whtuple).convert_alpha()
        pygame.draw.ellipse(self.alldata,fillcolor,self.alldata.get_rect(),0)
        self.setfill(0,0,0,0) #semi-transparent overlays on ellipses are supported
    def setfill(self,r,g,b,a=255): self.rfill = Var(r,0,255); self.gfill = Var(g,0,255); self.bfill = Var(b,0,255); self.afill = Var(a,0,255)
    """def __init__(self,filename,colorkey=None,cxstart=0,cystart=0,alphastart=255,whtuple=None):
        if filename == None: #for solids, pass None for filename, (R,G,B[,A]) for colorkey, and whtuple for (width,height)
            self.alldata = pygame.Surface(whtuple == None and (640,480) or whtuple).convert_alpha()
            (self.rfill,self.gfill,self.bfill,self.afill) = (silenVar(colorkey[0]),silenVar(colorkey[1]),silenVar(colorkey[2]),silenVar(len(colorkey) >= 4 and colorkey[3] or 0))
        else:
            self.alldata = load_image(filename,colorkey)
            (self.rfill,self.gfill,self.bfill,self.afill) = (silenVar(0),silenVar(0),silenVar(0),silenVar(0))
        self.allrect = self.alldata.get_rect()
        self.cx = silenVar(cxstart)
        self.cy = silenVar(cystart)
        self.xscale = silenVar(1.0) #these vars are proportions only, unfortunately
        self.yscale = silenVar(1.0)
        self.rot = silenVar(0.0)
        self.alpha = silenVar(alphastart) #only works with colorkey pictures
        self.fill = lambda: (self.rfill(),self.gfill(),self.bfill(),self.afill()) #afill only works on surfaces w/ alpha
        self.svlist = [self.cx,self.cy,self.xscale,self.yscale,self.rot,self.alpha,self.rfill,self.gfill,self.bfill,self.afill]
        self.oldxscale = 1.0
        self.oldyscale = 1.0
        self.oldrot = 0.0
        self.oldalpha = 255
        (self.oldrfill,self.oldgfill,self.oldbfill,self.oldafill) = self.oldfill = (0,0,0,0) #same ref, meaning changing one will affect the other (as it should)
        self.data = None
        self.rect = None
        self.pdelay = None
        self.oldpdelay = None
        self.changepic(True)"""
    def active(self): #if the sprite needs to be updated, this will return true.
        for var in self.svlist:
            if var.active(): return True
        return False
    def makeanim(self,times,seqargs,animargs=[],active=True): self.pframe = PicFrame(times,seqargs,animargs,active); self.changepic(True)
    def changepic(self,recalc=False): #graphics backend update. self.data is the surface to blit, and self.rect is its position relative to (self.cx,self.cy)
        rot = self.rot(); xs = self.xscale(); ys = self.yscale(); alpha = self.alpha(); fill = self.fill() #only do calcs once
        if self.pframe != None and self.oldpframe != self.pframe(): self.oldpframe = self.pframe(); recalc = True
        if rot != self.oldrot: self.oldrot = rot; recalc = True
        if xs != self.oldxscale or ys != self.oldyscale: self.oldxscale = xs; self.oldyscale = ys; recalc = True
        if alpha != self.oldalpha: self.oldalpha = alpha; recalc = True
        if fill != self.oldfill: (self.oldrfill,self.oldgfill,self.oldbfill,self.oldafill) = self.oldfill = fill; recalc = True
        if recalc: #surface update is necessary
            if self.pframe != None: self.oldpframe = self.pframe(); self.data = self.alldata.subsurface(self.pframe()).copy()
            else: self.oldpframe = None; self.data = self.alldata.copy()
            if xs < 0 or ys < 0: self.data = pygame.transform.flip(self.data,(xs<0),(ys<0))
            if abs(xs) != 1.0 or abs(ys)!= 1.0:
                temprect = self.data.get_rect()
                self.data = pygame.transform.scale(self.data,(int(abs(xs)*temprect.width),int(abs(ys)*temprect.height))) #scales by width,height
            if rot%360 != 0: self.data = pygame.transform.rotate(self.data,rot)
            if alpha != 255: self.data.set_alpha(alpha)
            if fill[3] != 0: tint = pygame.Surface(self.data.get_rect().size).convert_alpha(); tint.fill(fill); self.data.blit(tint,(0,0))
        self.rect = self.data.get_rect(center=(self.cx(),self.cy()))
    """
    def changepic(self,recalc=False): #after running, self.data will contain the surface to blit, and self.rect will contain its position respective to self.cx and self.cy
        (rot,rotflag) = (self.rot(),False)
        (xs,ys,scaleflag) = (self.xscale(),self.yscale(),False)
        (alpha,alphaflag) = (self.alpha(),False)
        (fill,fillflag) = (self.fill(),False)
        if self.pdelay != None and self.oldpdelay != self.pdelay(): (self.oldpdelay,recalc) = (self.pdelay(),True)
        if rot != self.oldrot: (self.oldrot,rotflag) = (rot,True)
        if xs != self.oldxscale or ys != self.oldyscale: (self.oldxscale,self.oldyscale,scaleflag) = (xs,ys,True)
        if alpha != self.oldalpha: (self.oldalpha,alphaflag) = (alpha,True)
        if fill != self.oldfill:
            (self.oldfill,fillflag) = (fill,True)
            (self.oldrfill,self.oldgfill,self.oldbfill,self.oldafill) = self.oldfill
        if fillflag:
            if self.pdelay != None: (self.oldpdelay,self.data) = (self.pdelay(),(self.alldata.subsurface(self.pdelay())).copy())
            else: (self.oldpdelay,self.data) = (None,self.alldata.copy())
            self.data.fill(fill)
            if rotflag or scaleflag: #these flags could still have significance in a fill surface
                if xs < 0 or ys < 0: self.data = pygame.transform.flip(self.data,(xs<0),(ys<0))
                if abs(xs) != 1.0 or abs(ys)!= 1.0:
                    temprect = self.data.get_rect()
                    self.data = pygame.transform.scale(self.data,(int(abs(xs)*temprect.width),int(abs(ys)*temprect.height))) #scales by width,height
                if rot%360 != 0: self.data = pygame.transform.rotate(self.data,rot)
        elif rotflag or scaleflag or alphaflag or recalc: #surface update is necessary
            if self.pdelay != None: (self.oldpdelay,self.data) = (self.pdelay(),(self.alldata.subsurface(self.pdelay())).copy())
            else: (self.oldpdelay,self.data) = (None,self.alldata.copy())
            if xs < 0 or ys < 0: self.data = pygame.transform.flip(self.data,(xs<0),(ys<0))
            if abs(xs) != 1.0 or abs(ys)!= 1.0:
                temprect = self.data.get_rect()
                self.data = pygame.transform.scale(self.data,(int(abs(xs)*temprect.width),int(abs(ys)*temprect.height))) #scales by width,height
            if rot%360 != 0: self.data = pygame.transform.rotate(self.data,rot)
            if alpha != 255: self.data.set_alpha(alpha)
        temprect = self.data.get_rect()
        self.rect = Rect(self.cx() - temprect.w/2, self.cy() - temprect.h/2, temprect.w, temprect.h)"""
    def settimestep(self,timestep=1.0): self.__class__.timestep = timestep
    def update(self,timestep=None,recalc=False):
        if timestep == None: timestep = self.__class__.timestep #default class timestep. the 'world' time, so to speak...
        for var in self.svlist: var.update(timestep)
        if self.pframe != None: self.pframe.update(timestep)
        rot = self.rot()
        if rot < 0 or rot >= 360: self.rot.offset -= rot//360*360 #rotfcn = lambda x: x-int(floor(x))/360*360
        self.changepic(recalc)
    def __call__(self): return self.data #should always contain the graphic to draw
    def area_rect(self): return self.pdelay != None and self.pdelay() or self.alldata.get_rect() #the area rect in self.alldata, NOT self.data (don't use when blitting)
    def endtint(self): self.rfill.end(); self.gfill.end(); self.bfill.end(); self.afill.end()
    def end(self): #ends all vars, and clears their anim lists
        for var in self.svlist: var.end()
    def setpos(self,cx,cy): self.cx.offset = cx; self.cy.offset = cy
    def addpos(self,cx,cy): self.cx.offset += cx; self.cy.offset += cy
    def move(self,cxamt,cyamt,posargs,animtype=0,ctrlargs=[],loopargs=[]):
        rtn = self.cx.add(posargs,cxamt,animtype,ctrlargs,loopargs) #checks are in Var, so no motion is handled (returns None)
        return self.cy.add(posargs,cyamt,animtype,ctrlargs,loopargs) or rtn #returns None if both are None, otherwise a valid AnimVar
    def moveto(self,cxpos,cypos,posargs,animtype=0,ctrlargs=[],loopargs=[]):
        rtn = self.cx.movto(posargs,cxpos,animtype,ctrlargs,loopargs)
        return self.cy.movto(posargs,cypos,animtype,ctrlargs,loopargs) or rtn
    def rotate(self,cwdegs,posargs,animtype=0,ctrlargs=[],loopargs=[]): return self.rot.add(posargs,cwdegs,animtype,ctrlargs,loopargs)
    def rotto(self,cwdegpos,posargs,animtype=0,ctrlargs=[],loopargs=[]): return self.rot.movto(posargs,cwdegpos,animtype,ctrlargs,loopargs)
    def scale(self,xprop,yprop,posargs,animtype=0,ctrlargs=[],loopargs=[]):
        rtn = self.xscale.movto(posargs,xprop,animtype,ctrlargs,loopargs)
        return self.yscale.movto(posargs,yprop,animtype,ctrlargs,loopargs) or rtn
    def addscale(self,xpamt,ypamt,posargs,animtype=0,ctrlargs=[],loopargs=[]):
        rtn = self.xscale.add(posargs,xpamt,animtype,ctrlargs,loopargs)
        return self.yscale.add(posargs,ypamt,animtype,ctrlargs,loopargs) or rtn
    def tint(self,fill,posargs,animtype=0,ctrlargs=[],loopargs=[]):
        rtn = (fill[0] != None) and self.rfill.movto(posargs,fill[0],animtype,ctrlargs,loopargs) or None
        rtn = (fill[1] != None) and self.gfill.movto(posargs,fill[1],animtype,ctrlargs,loopargs) or rtn
        rtn = (fill[2] != None) and self.bfill.movto(posargs,fill[2],animtype,ctrlargs,loopargs) or rtn
        return (len(fill) >= 4 and fill[3] != None) and self.afill.movto(posargs,fill[3],animtype,ctrlargs,loopargs) or rtn
    def addtint(self,fillamt,posargs,animtype=0,ctrlargs=[],loopargs=[]): #adds to tint (fill AnimVars are safe due to clipping mins/maxs)
        rtn = (fillamt[0] != None) and self.rfill.add(posargs,fillamt[0],animtype,ctrlargs,loopargs) or None
        rtn = (fillamt[1] != None) and self.gfill.add(posargs,fillamt[1],animtype,ctrlargs,loopargs) or rtn
        rtn = (fillamt[2] != None) and self.bfill.add(posargs,fillamt[2],animtype,ctrlargs,loopargs) or rtn
        return (len(fillamt) >= 4 and fillamt[3] != None) and self.afill.add(posargs,fillamt[3],animtype,ctrlargs,loopargs) or rtn
    def fadein(self,posargs,ctrlargs=[],loopargs=[]): return self.alpha.movto(posargs,255,0,ctrlargs,loopargs)
    def fadeout(self,posargs,ctrlargs=[],loopargs=[]): return self.alpha.movto(posargs,0,0,ctrlargs,loopargs)
if __name__ == '__main__':
    print CoreVar(10,100,6).__doc__
    print LinCtr([10]).__doc__
    print AnimVar([10],100,6,[],[0.0,None]).__doc__
    print Var(0,ubound=255).__doc__
    print PicFrame([5,10,15,20,25,30,35,40],[(100,50),4,2,True],[2,1,4,2],True).__doc__
    print Sprite.__doc__
    pygame.init()
    pygame.display.set_caption('silen.Sprite Tester')
    pygame.mouse.set_visible(0)
    screen = pygame.display.set_mode((640,480))
    sprite = Sprite(0,["fwcircle.png",(255,255,255),0],(320,240))
    sprite.fadein([30],[True,True])
    from event import *
    fsflag = False; event = silenEvent(); clock = pygame.time.Clock()
    #tint = pygame.Surface((216,108)).convert_alpha(); tint.fill((255,0,0,50))
    #sprite.blit(tint,(0,0))#,None,pygame.BLEND_SUB)
    while True:
        event.update()
        if event.fsflag:
            if fsflag: screen = pygame.display.set_mode((640,480))
            else: screen = pygame.display.set_mode((640,480),pygame.FULLSCREEN|pygame.DOUBLEBUF|pygame.HWSURFACE|pygame.NOFRAME)
            fsflag = not fsflag; event.fsflag = False
        screen.fill((0,0,0))
        sprite.update()
        #print sprite.alpha(), sprite.alpha.rawfinal()
        screen.blit(sprite.data,sprite.rect)
        pygame.display.flip()
        clock.tick(30)
    #test = LinCtr([10,0,0,1,0,0,0,0,0,0,0][:4])
    #for i in range(31): test.update(); print test()
    #print len(None)#fails (TypeError)
